<?php 

class ReportesModel
{
    private $conn;

    public function __construct()
    {
        require_once __DIR__ . '/../../../config/database.php';
        $db = new Database();
        $this->conn = $db->getConnection();
    }


    public function informacionProductoraEventos(int $idProductora): array
    {
        // 1) Traer info de la productora
        $sqlProd = "
            SELECT 
                idProductora,
                nombreProductora,
                logoProductora,
                informacionProductora,
                fechaInicioProductora,
                estadoProductora
            FROM productora
            WHERE idProductora = :idProductora
            LIMIT 1
        ";
    
        $stmt = $this->conn->prepare($sqlProd);
        $stmt->bindValue(':idProductora', $idProductora, PDO::PARAM_INT);
        $stmt->execute();
        $prod = $stmt->fetch(PDO::FETCH_ASSOC);
    
        if (!$prod) {
            // Si no existe la productora, devolvemos estructura vacía
            return [
                'idProductora'     => $idProductora,
                'nombreProductora' => null,
                'eventos'          => []
            ];
        }
    
        // 2) Traer eventos + localización + agregados de entradas
        $sqlEventos = "
            SELECT 
                p.idProductora,
                p.nombreProductora,
                p.logoProductora,
    
                e.idEvento,
                e.tituloEvento,
                e.logoEvento,
                e.codigo,
                e.llave,
                e.diaEvento,
                e.HorainicioEvento AS horaInicioEvento,
    
                le.paisEvento,
                le.lugarEvento,
    
                COALESCE(SUM(ee.cantidadEntradas), 0) AS capacidadTotal,
                COALESCE(SUM(ee.precioEntrada * ee.cantidadEntradas), 0) AS montoPotencial
            FROM productora p
            INNER JOIN evento e 
                ON e.idProductora = p.idProductora
            LEFT JOIN localizacionEvento le 
                ON le.idEvento = e.idEvento
            LEFT JOIN entradasEvento ee 
                ON ee.idEvento = e.idEvento
            WHERE p.idProductora = :idProductora
            GROUP BY 
                p.idProductora,
                p.nombreProductora,
                p.logoProductora,
                e.idEvento,
                e.tituloEvento,
                e.logoEvento,
                e.codigo,
                e.llave,
                e.diaEvento,
                e.HorainicioEvento,
                le.paisEvento,
                le.lugarEvento
            ORDER BY e.diaEvento, e.HorainicioEvento
        ";
    
        $stmt = $this->conn->prepare($sqlEventos);
        $stmt->bindValue(':idProductora', $idProductora, PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
        $eventos = [];
        foreach ($rows as $r) {
            $eventos[] = [
                'idEvento'         => (int)$r['idEvento'],
                'tituloEvento'     => $r['tituloEvento'],
                'logoEvento'     => $r['logoEvento'],
                'codigo'           => $r['codigo'],
                'llave'            => $r['llave'],
                'diaEvento'        => $r['diaEvento'],
                'horaInicioEvento' => $r['horaInicioEvento'],
    
                'paisEvento'       => $r['paisEvento'],
                'lugarEvento'      => $r['lugarEvento'],
    
                // suma de cantidadEntradas por evento
                'capacidadTotal'   => (int)$r['capacidadTotal'],
    
                // suma de (precioEntrada * cantidadEntradas) por evento
                'montoPotencial'   => (float)$r['montoPotencial'],
            ];
        }
    
        return [
            'idProductora'             => (int)$prod['idProductora'],
            'nombreProductora'         => $prod['nombreProductora'],
            'logoProductora'           => $prod['logoProductora'],
            'informacionProductora'    => $prod['informacionProductora'],
            'fechaInicioProductora'    => $prod['fechaInicioProductora'],
            'estadoProductora'         => (int)$prod['estadoProductora'],
            'eventos'                  => $eventos
        ];
    }


    private function rangoFechasSql(?string $desde, ?string $hasta): string
    {
        $cond = "";
        if ($desde) {
            $cond .= " AND v.fechaVenta >= :desde ";
        }
        if ($hasta) {
            $cond .= " AND v.fechaVenta <= :hasta ";
        }
        return $cond;
    }

    public function kpiProductora(int $idProductora, ?string $desde, ?string $hasta): array
    {
        $fechaCond = $this->rangoFechasSql($desde, $hasta);

        // 1) Totales de venta
        $sqlTotales = "
            SELECT 
                COALESCE(SUM(v.total), 0)       AS totalVentasBrutas,
                COALESCE(SUM(v.totalNeto), 0)   AS totalNeto,
                COALESCE(SUM(v.totalIva), 0)    AS totalIva,
                COUNT(*)                        AS cantidadVentas
            FROM ventas v
            INNER JOIN ventaDetalle vd ON vd.ventaId = v.id
            INNER JOIN evento e        ON e.idEvento = vd.idEvento
            WHERE e.idProductora = :idProductora
              AND v.estadoVenta = 'pagada'
              $fechaCond
        ";
        $stmt = $this->conn->prepare($sqlTotales);
        if ($desde) $stmt->bindValue(':desde', $desde);
        if ($hasta) $stmt->bindValue(':hasta', $hasta);
        $stmt->bindValue(':idProductora', $idProductora, PDO::PARAM_INT);
        $stmt->execute();
        $totales = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];

        // 2) Tickets vendidos y capacidad
        $sqlTickets = "
            SELECT 
                COALESCE(SUM(vd.cantidad), 0) AS ticketsVendidos,
                COALESCE(SUM(ee.cantidadEntradas), 0) AS capacidadTotal
            FROM ventas v
            INNER JOIN ventaDetalle vd   ON vd.ventaId = v.id
            INNER JOIN entradasEvento ee ON ee.idEntradaEvento = vd.idEntradaEvento
            INNER JOIN evento e          ON e.idEvento = vd.idEvento
            WHERE e.idProductora = :idProductora
              AND v.estadoVenta = 'pagada'
              $fechaCond
        ";
        $stmt = $this->conn->prepare($sqlTickets);
        if ($desde) $stmt->bindValue(':desde', $desde);
        if ($hasta) $stmt->bindValue(':hasta', $hasta);
        $stmt->bindValue(':idProductora', $idProductora, PDO::PARAM_INT);
        $stmt->execute();
        $tickets = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];

        // 3) Compradores únicos
        $sqlCompradores = "
            SELECT COUNT(DISTINCT v.usuarioId) AS compradoresUnicos
            FROM ventas v
            INNER JOIN ventaDetalle vd ON vd.ventaId = v.id
            INNER JOIN evento e        ON e.idEvento = vd.idEvento
            WHERE e.idProductora = :idProductora
              AND v.estadoVenta = 'pagada'
              $fechaCond
        ";
        $stmt = $this->conn->prepare($sqlCompradores);
        if ($desde) $stmt->bindValue(':desde', $desde);
        if ($hasta) $stmt->bindValue(':hasta', $hasta);
        $stmt->bindValue(':idProductora', $idProductora, PDO::PARAM_INT);
        $stmt->execute();
        $compradores = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];

        // 4) Cortesías enviadas
        $sqlCortesias = "
            SELECT COUNT(*) AS cantidadCortesias
            FROM cuponDescuento cd
            INNER JOIN cuponAsignado ca ON ca.idCupon = cd.idCupon
            INNER JOIN evento e         ON e.idEvento = cd.idEvento
            WHERE e.idProductora = :idProductora
              AND cd.tipoDescuento = 'cortesia'
        ";
        $stmt = $this->conn->prepare($sqlCortesias);
        $stmt->execute([':idProductora' => $idProductora]);
        $cortesias = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];

        // 5) Ventas por medio de pago
        $sqlMedios = "
            SELECT 
                mp.nombre,
                COALESCE(SUM(v.total), 0) AS totalPorMedio,
                COUNT(*)                  AS cantidadVentas
            FROM ventas v
            INNER JOIN pagos p   ON p.ventaId    = v.id
            INNER JOIN medioPago mp ON mp.id     = p.medioPagoId
            INNER JOIN ventaDetalle vd ON vd.ventaId = v.id
            INNER JOIN evento e  ON e.idEvento   = vd.idEvento
            WHERE e.idProductora = :idProductora
              AND v.estadoVenta = 'pagada'
              $fechaCond
            GROUP BY mp.id, mp.nombre
        ";
        $stmt = $this->conn->prepare($sqlMedios);
        if ($desde) $stmt->bindValue(':desde', $desde);
        if ($hasta) $stmt->bindValue(':hasta', $hasta);
        $stmt->bindValue(':idProductora', $idProductora, PDO::PARAM_INT);
        $stmt->execute();
        $medios = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Armar respuesta
        $ticketsVendidos = (int)($tickets['ticketsVendidos'] ?? 0);
        $capacidadTotal  = (int)($tickets['capacidadTotal'] ?? 0);
        $porcOcupacion   = $capacidadTotal > 0 ? round(($ticketsVendidos / $capacidadTotal) * 100, 2) : 0;

        $totalVentasBrutas = (float)($totales['totalVentasBrutas'] ?? 0);
        $cantidadVentas    = (int)($totales['cantidadVentas'] ?? 0);
        $promedioTicket    = $ticketsVendidos > 0 ? round($totalVentasBrutas / $ticketsVendidos, 2) : 0;

        return [
            'totalVentasBrutas'   => $totalVentasBrutas,
            'totalNeto'           => (float)($totales['totalNeto'] ?? 0),
            'totalIva'            => (float)($totales['totalIva'] ?? 0),
            'cantidadVentas'      => $cantidadVentas,
            'ticketsVendidos'     => $ticketsVendidos,
            'capacidadTotal'      => $capacidadTotal,
            'porcentajeOcupacion' => $porcOcupacion,
            'compradoresUnicos'   => (int)($compradores['compradoresUnicos'] ?? 0),
            'promedioValorTicket' => $promedioTicket,
            'cortesiasEnviadas'   => (int)($cortesias['cantidadCortesias'] ?? 0),
            'ventasPorMedioPago'  => $medios
        ];
    }

    /**
     * Informes parametrizados
     * $params = ['idProductora','tipo','desde','hasta']
     */
    public function informeProductora(array $params): array
    {
        $idProductora = (int)$params['idProductora'];
        $tipo         = $params['tipo'] ?? 'ventas';
        $desde        = $params['desde'] ?? null;
        $hasta        = $params['hasta'] ?? null;

        $fechaCond = $this->rangoFechasSql($desde, $hasta);

        switch ($tipo) {
            case 'ventas':
                $sql = "
                    SELECT 
                        v.id,
                        v.fechaVenta,
                        v.totalNeto,
                        v.totalIva,
                        v.total,
                        v.estadoVenta,
                        u.nombre,
                        u.apellido,
                        u.correo
                    FROM ventas v
                    INNER JOIN usuarios u ON u.id = v.usuarioId
                    INNER JOIN ventaDetalle vd ON vd.ventaId = v.id
                    INNER JOIN evento e ON e.idEvento = vd.idEvento
                    WHERE e.idProductora = :idProductora
                      $fechaCond
                    GROUP BY v.id
                    ORDER BY v.fechaVenta DESC
                ";
                break;

            case 'tickets':
                $sql = "
                    SELECT 
                        e.idEvento,
                        e.tituloEvento,
                        vd.idEntradaEvento,
                        vd.tituloEntrada,
                        SUM(vd.cantidad) AS ticketsVendidos,
                        ee.cantidadEntradas AS stockTotal
                    FROM ventaDetalle vd
                    INNER JOIN ventas v ON v.id = vd.ventaId
                    INNER JOIN evento e ON e.idEvento = vd.idEvento
                    INNER JOIN entradasEvento ee ON ee.idEntradaEvento = vd.idEntradaEvento
                    WHERE e.idProductora = :idProductora
                      $fechaCond
                    GROUP BY e.idEvento, vd.idEntradaEvento
                    ORDER BY e.idEvento ASC, vd.idEntradaEvento ASC
                ";
                break;

            case 'asistentes':
                $sql = "
                    SELECT 
                        e.idEvento,
                        e.tituloEvento,
                        u.id         AS usuarioId,
                        u.nombre,
                        u.apellido,
                        u.correo,
                        u.rut,
                        SUM(vd.cantidad) AS ticketsComprados
                    FROM ventas v
                    INNER JOIN usuarios u ON u.id = v.usuarioId
                    INNER JOIN ventaDetalle vd ON vd.ventaId = v.id
                    INNER JOIN evento e ON e.idEvento = vd.idEvento
                    WHERE e.idProductora = :idProductora
                      AND v.estadoVenta = 'pagada'
                      $fechaCond
                    GROUP BY e.idEvento, u.id
                    ORDER BY e.idEvento ASC, u.nombre ASC
                ";
                break;

            case 'consumos':
                $sql = "
                    SELECT 
                        e.idEvento,
                        e.tituloEvento,
                        ce.nombreConsumo,
                        SUM(vd.cantidad) AS cantidadVendida
                    FROM ventas v
                    INNER JOIN ventaDetalle vd ON vd.ventaId = v.id
                    INNER JOIN evento e ON e.idEvento = vd.idEvento
                    INNER JOIN consumoEvento ce ON ce.idEvento = e.idEvento
                    WHERE e.idProductora = :idProductora
                      $fechaCond
                    GROUP BY e.idEvento, ce.idConsumoEvento
                ";
                break;

            case 'conversion':
                // conversión básica: ventas vs usuarios que agregaron algo al carro
                $sql = "
                    SELECT 
                        e.idEvento,
                        e.tituloEvento,
                        COUNT(DISTINCT v.usuarioId)     AS compradores,
                        COUNT(DISTINCT tc.usuarioId)    AS usuariosCarro
                    FROM evento e
                    LEFT JOIN ventaDetalle vd ON vd.idEvento = e.idEvento
                    LEFT JOIN ventas v        ON v.id = vd.ventaId
                    LEFT JOIN transaccionesCarro tc ON tc.idEvento = e.idEvento
                    WHERE e.idProductora = :idProductora
                      $fechaCond
                    GROUP BY e.idEvento
                ";
                break;

            default:
                // por si viene algo raro
                return [];
        }

        $stmt = $this->conn->prepare($sql);
        if ($desde) $stmt->bindValue(':desde', $desde);
        if ($hasta) $stmt->bindValue(':hasta', $hasta);
        $stmt->bindValue(':idProductora', $idProductora, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
