<?php
declare(strict_types=1);

require_once __DIR__ . '/AuthController.php';
require_once __DIR__ . '/../models/empleo/bolsaEmpleo.php';

class EmpleoController extends AuthController
{
    /* ========= PUBLICO / CLIENTE ========= */

    public function listarEmpleosPublicos()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true) ?? [];
        $ciudad = $data['ciudad'] ?? null;
        $tipoEmpleo = $data['tipoEmpleo'] ?? null;

        $model = new Empleos();
        $items = $model->listarOfertasPublicas($ciudad, $tipoEmpleo);

        echo json_encode(["status"=>"ok","message"=>$items]);
    }

    public function detalleEmpleo()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);
        if (!isset($data['idEmpleo'])) {
            http_response_code(400);
            echo json_encode(["status"=>"error","message"=>"idEmpleo es requerido"]);
            return;
        }

        $model = new Empleos();
        $item = $model->obtenerOferta((int)$data['idEmpleo']);

        if (!$item) {
            http_response_code(404);
            echo json_encode(["status"=>"error","message"=>"Empleo no existe"]);
            return;
        }

        echo json_encode(["status"=>"ok","message"=>$item]);
    }

    public function postularEmpleo()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);
        if (!isset($data['usuarioId'], $data['idEmpleo'], $data['mensajeInicial'])) {
            http_response_code(400);
            echo json_encode(["status"=>"error","message"=>"Parámetros incompletos"]);
            return;
        }

        $model = new Empleos();
        $resp = $model->postular(
            (int)$data['usuarioId'],
            (int)$data['idEmpleo'],
            (string)$data['mensajeInicial'],
            $data['cvUrl'] ?? null,
            $data['respuestasJson'] ?? null
        );

        if ($resp['ok']) {
            echo json_encode(["status"=>"ok","message"=>"Postulación creada","data"=>$resp]);
            return;
        }

        http_response_code(409);
        echo json_encode(["status"=>"error","message"=>$resp['error'],"data"=>$resp]);
    }

    public function enviarMensajeUsuario()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);
        if (!isset($data['usuarioId'], $data['idEmpleo'], $data['mensaje'])) {
            http_response_code(400);
            echo json_encode(["status"=>"error","message"=>"Parámetros incompletos"]);
            return;
        }

        $model = new Empleos();
        $resp = $model->enviarMensajeUsuario((int)$data['usuarioId'], (int)$data['idEmpleo'], (string)$data['mensaje']);

        if ($resp['ok']) {
            echo json_encode(["status"=>"ok","message"=>"Mensaje enviado","data"=>$resp]);
            return;
        }

        http_response_code(500);
        echo json_encode(["status"=>"error","message"=>$resp['error'] ?? 'No se pudo enviar']);
    }

    public function listarMensajes()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);
        if (!isset($data['idEmpleo'])) {
            http_response_code(400);
            echo json_encode(["status"=>"error","message"=>"idEmpleo es requerido"]);
            return;
        }

        $model = new Empleos();
        $items = $model->listarMensajes((int)$data['idEmpleo']);

        echo json_encode(["status"=>"ok","message"=>$items]);
    }

    public function calificarEmpleo()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);
        $required = ['idEmpleo','calificadorTipo','calificadoTipo','estrellas'];
        foreach ($required as $r) {
            if (!isset($data[$r])) {
                http_response_code(400);
                echo json_encode(["status"=>"error","message"=>"Falta: {$r}"]);
                return;
            }
        }

        $model = new Empleos();
        $resp = $model->calificar($data);

        if ($resp['ok']) {
            echo json_encode(["status"=>"ok","message"=>"Calificación guardada","data"=>$resp]);
            return;
        }

        http_response_code(500);
        echo json_encode(["status"=>"error","message"=>$resp['error'] ?? 'No se pudo calificar']);
    }

    /* ========= PRODUCTORA ========= */

    public function listarEmpleosProductora()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);
        if (!isset($data['idProductora'])) {
            http_response_code(400);
            echo json_encode(["status"=>"error","message"=>"idProductora es requerido"]);
            return;
        }

        $model = new Empleos();
        $items = $model->listarOfertasProductora((int)$data['idProductora']);

        echo json_encode(["status"=>"ok","message"=>$items]);
    }

    public function crearEmpleo()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);
        $required = ['idProductora','cargo','acuerdo','tipoEmpleo','remuneracion','fechaInicioTrabajo'];
        foreach ($required as $r) {
            if (!isset($data[$r])) {
                http_response_code(400);
                echo json_encode(["status"=>"error","message"=>"Falta: {$r}"]);
                return;
            }
        }

        $model = new Empleos();
        $resp = $model->crearOferta((int)$data['idProductora'], $data);

        if ($resp['ok']) {
            echo json_encode(["status"=>"ok","message"=>"Empleo creado","data"=>$resp]);
            return;
        }

        http_response_code(500);
        echo json_encode(["status"=>"error","message"=>$resp['error'] ?? 'No se pudo crear']);
    }

    public function editarEmpleo()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);
        if (!isset($data['idProductora'], $data['idEmpleo'])) {
            http_response_code(400);
            echo json_encode(["status"=>"error","message"=>"idProductora e idEmpleo son requeridos"]);
            return;
        }

        $model = new Empleos();
        $resp = $model->editarOferta((int)$data['idProductora'], (int)$data['idEmpleo'], $data);

        if ($resp['ok']) {
            echo json_encode(["status"=>"ok","message"=>"Empleo editado"]);
            return;
        }

        http_response_code(409);
        echo json_encode(["status"=>"error","message"=>$resp['error'],"data"=>$resp]);
    }

    public function desactivarEmpleo()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);
        if (!isset($data['idProductora'], $data['idEmpleo'])) {
            http_response_code(400);
            echo json_encode(["status"=>"error","message"=>"idProductora e idEmpleo son requeridos"]);
            return;
        }

        $model = new Empleos();
        $resp = $model->desactivarOferta((int)$data['idProductora'], (int)$data['idEmpleo']);

        if ($resp['ok']) {
            echo json_encode(["status"=>"ok","message"=>"Empleo desactivado"]);
            return;
        }

        http_response_code(409);
        echo json_encode(["status"=>"error","message"=>$resp['error'],"data"=>$resp]);
    }

    public function verPostulantes()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);
        if (!isset($data['idProductora'], $data['idEmpleo'])) {
            http_response_code(400);
            echo json_encode(["status"=>"error","message"=>"idProductora e idEmpleo son requeridos"]);
            return;
        }

        $model = new Empleos();
        $resp = $model->listarPostulantes((int)$data['idProductora'], (int)$data['idEmpleo']);

        if ($resp['ok']) {
            echo json_encode(["status"=>"ok","message"=>$resp['items']]);
            return;
        }

        http_response_code(409);
        echo json_encode(["status"=>"error","message"=>$resp['error']]);
    }

    public function cambiarEstadoPostulacion()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);
        if (!isset($data['idProductora'], $data['idPostulacion'], $data['estado'])) {
            http_response_code(400);
            echo json_encode(["status"=>"error","message"=>"Parámetros incompletos"]);
            return;
        }

        $model = new Empleos();
        $resp = $model->cambiarEstadoPostulacion((int)$data['idProductora'], (int)$data['idPostulacion'], (string)$data['estado']);

        if ($resp['ok']) {
            echo json_encode(["status"=>"ok","message"=>"Estado actualizado"]);
            return;
        }

        http_response_code(409);
        echo json_encode(["status"=>"error","message"=>$resp['error']]);
    }

    public function enviarMensajeProductora()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);
        if (!isset($data['idProductora'], $data['idEmpleo'], $data['mensaje'])) {
            http_response_code(400);
            echo json_encode(["status"=>"error","message"=>"Parámetros incompletos"]);
            return;
        }

        $model = new Empleos();
        $resp = $model->enviarMensajeProductora((int)$data['idProductora'], (int)$data['idEmpleo'], (string)$data['mensaje']);

        if ($resp['ok']) {
            echo json_encode(["status"=>"ok","message"=>"Mensaje enviado","data"=>$resp]);
            return;
        }

        http_response_code(409);
        echo json_encode(["status"=>"error","message"=>$resp['error']]);
    }
}
